# finaquant Financial Analytics - www.finaquant.com
# Copyright: Tunc Ali Ktkcoglu 2012, version: 24April2012
# All functions required for Share Class Fee Calculater
# Related web page:
# http://finaquant.com/fee-calculator-and-analyzer-for-share-classes-of-mutual-funds-1-calculation/991

# ***************************************************************************
# help function for displaying variables
# ***************************************************************************
display_variable = function(v, vname) { 
print(paste(vname,' ='),quote = FALSE)
print(v,quote = FALSE)
}

# ***************************************************************************
# Calculate effective fee rate to be applied on the whole asset volume
# ***************************************************************************
calculate_fee_scale = function(AssetVolume, FeeScale, ScaleCalcLogic){
	# check size of FeeRates
	rows = nrow(FeeScale)
	cols = ncol(FeeScale)

	if (cols != 2) stop('Matrix FeeRates must have 2 columns!')

	# find matching volume range
	VolumeRange = 1
	while (VolumeRange < rows && AssetVolume > FeeScale[VolumeRange, 2]) VolumeRange = VolumeRange + 1

	if (VolumeRange > rows) stop('AssetVolume cannot be the larger than the highest volume limit in fee scale!')

	if (ScaleCalcLogic == 'class') {
		EffectiveFeeRate = FeeScale[VolumeRange,1]
	} else if (ScaleCalcLogic == 'level'){
		TotalFee = 0;
		PreviousRangeLimit = 0;
			
		for (i in 1:VolumeRange){
			if (i < VolumeRange)
				RangeVolume = FeeScale[i,2] - PreviousRangeLimit
			else    # i = VolumeRange
				RangeVolume = AssetVolume - PreviousRangeLimit

			TotalFee = TotalFee + FeeScale[i,1] * RangeVolume
			PreviousRangeLimit = FeeScale[i,2]
		}  # for
		EffectiveFeeRate = TotalFee / AssetVolume
	} else {
		stop('Undefined fee scale calculation logic!')
	}  	# if
return(list(EffectiveFeeRate, VolumeRange))
}	# function

# ***************************************************************************
# single-period function; calculates all fees for a single period
# ***************************************************************************
FeeCalculator_SinglePeriod = function( 
    FrondEndLoadRates,
    BackEndLoadCDSCRates,
    AnnualDistributionFeeRate,
    AnnualManagementFeeRates,
    MngFeeScaleLogic,
    OtherAnnualServiceFeeRates,
    OtherSubscriptionFeeRates,
    OtherRedemptionFeeRates,
    AssumedFundReturn,
    AssetVolume,
    NumShares,     			
    CurrentPeriod,                 
    N) 	{    				

	# initiate result vector for storing all results
	ResultVector = rep(0, 13)

	# front-end load
	if (CurrentPeriod == 1){
		res = calculate_fee_scale(AssetVolume, FrondEndLoadRates, 'class')
		EffectiveFeeRate = res[[1]]
		FrondEndLoad = EffectiveFeeRate * AssetVolume / 100
	} else {
		FrondEndLoad = 0
	}
	ResultVector[1] = FrondEndLoad
	
	# other subscription fees
	if (CurrentPeriod == 1)
		OtherSubscriptionFees = OtherSubscriptionFeeRates[1] * AssetVolume / 100 + OtherSubscriptionFeeRates[2]* NumShares
	else
		OtherSubscriptionFees = 0
	ResultVector[2] = OtherSubscriptionFees

	# Remaining asset value after front-end and other subscription fees
	AssetVolumeAfterSubscriptionFees = AssetVolume - FrondEndLoad - OtherSubscriptionFees
	ResultVector[3] = AssetVolumeAfterSubscriptionFees

	# Period-end asset value before management and service fees
	ResultVector[4] = AssumedFundReturn
	AssetVolumeBeforeManagementFees = AssetVolumeAfterSubscriptionFees * (1 + AssumedFundReturn/100)
	ResultVector[5] = AssetVolumeBeforeManagementFees

	# Management fees in %
	res = calculate_fee_scale(AssetVolumeBeforeManagementFees, AnnualManagementFeeRates, MngFeeScaleLogic)
	EffectiveMngFeeRate = res[[1]]
	ResultVector[6] = EffectiveMngFeeRate

	ManagementFee = EffectiveMngFeeRate * AssetVolumeBeforeManagementFees / 10000 	# mng fee rate in BP
	ResultVector[7] = ManagementFee

	# Annual distribution fees in %
	AnnualistributionFees = AnnualDistributionFeeRate * AssetVolumeBeforeManagementFees / 100
	ResultVector[8] = AnnualistributionFees

	# Other annual service fees in BP
	AnnualServiceFees = OtherAnnualServiceFeeRates[1] * AssetVolumeBeforeManagementFees / 10000 + OtherAnnualServiceFeeRates[2] * NumShares
	ResultVector[9] = AnnualServiceFees

	# Asset value before redemption fees
	AssetVolumeBeforeRedemptionFees = AssetVolumeBeforeManagementFees - ManagementFee - AnnualistributionFees - AnnualServiceFees
	ResultVector[10] = AssetVolumeBeforeRedemptionFees

	# Back-end load with CDSC
	if (N == CurrentPeriod && N <= length(BackEndLoadCDSCRates))
		BackEndLoadCDSC = BackEndLoadCDSCRates[N] * AssetVolumeBeforeRedemptionFees / 100
	else
		BackEndLoadCDSC = 0

	ResultVector[11] = BackEndLoadCDSC

	# Other redemption fees
	if (N == CurrentPeriod)
		OtherRedemptionFees = OtherRedemptionFeeRates[1] * AssetVolumeBeforeRedemptionFees / 100 + OtherRedemptionFeeRates[2] * NumShares
	else
		OtherRedemptionFees = 0

	ResultVector[12] = OtherRedemptionFees

	# Asset value after redemption fees
	AssetVolumeAfterRedemptionFees = AssetVolumeBeforeRedemptionFees - BackEndLoadCDSC - OtherRedemptionFees
	ResultVector[13] = AssetVolumeAfterRedemptionFees
	return(ResultVector)
}	# function

# ***************************************************************************
# multi-period function; Calculates fees for N calculation periods
# ***************************************************************************
FeeCalculator_MultiPeriod = function(
	FrondEndLoadRates,              
    BackEndLoadCDSCRates,           
    AnnualDistributionFeeRate,      
    AnnualManagementFeeRates,       
    MngFeeScaleLogic,               
    OtherAnnualServiceFeeRates,     
    OtherSubscriptionFeeRates,      
    OtherRedemptionFeeRates,        
    AssumedFundReturns,             
    SharePrice,                     # gross subscription price
    NumShares,                      # number of shares
    N)                             	# total number of periods (f.e. years, quarters etc.)
{
	# Fields in result table
	ResultMatrixFields = c('FrondEndLoad', 'OtherSubsFees', 'AVafterSubsFees', 
		'AssumedFundReturn','AVbeforeMngFees','EffMngFeeRate','ManagementFee', 
		'AnnualDistrFee','AnnualServiceFee','AVbeforeRedempFee','BackEndLoadCDSC', 
		'OtherRedempFee','AVAfterRedempFee')

	# initiate result matrices and vectors
	ResultMatrix = matrix(0, nrow=N, ncol=13)
	EffectiveFundReturns = rep(0,N)
	EffectiveAssetValue = rep(0,N)

	# prepare required input parameters for the single-period fee function
	AssetVolume = SharePrice * NumShares 	# initial asset volume

	for (CurrentPeriod in 1:N){
		AssumedFundReturn = AssumedFundReturns[CurrentPeriod]
		
		# call single-period fee function
		ResultVector = FeeCalculator_SinglePeriod(
		FrondEndLoadRates,              
		BackEndLoadCDSCRates,           
		AnnualDistributionFeeRate,      
		AnnualManagementFeeRates,       
		MngFeeScaleLogic,               
		OtherAnnualServiceFeeRates,     
		OtherSubscriptionFeeRates,      
		OtherRedemptionFeeRates,        
		AssumedFundReturn,              
		AssetVolume,                    
		NumShares,             	# number of shares
		CurrentPeriod,                  
		N)      				# total number of calculation periods
		
		# store results
		ResultMatrix[CurrentPeriod,] = ResultVector
		EffectiveAssetValue[CurrentPeriod] = ResultVector[length(ResultVector)]
		EffectiveFundReturns[CurrentPeriod] = (EffectiveAssetValue[CurrentPeriod] - AssetVolume) /  AssetVolume * 100
		AssetVolume = EffectiveAssetValue[CurrentPeriod] 	# asset volume as input for the next period
	}	# for
return(list(ResultMatrix,ResultMatrixFields,EffectiveFundReturns,EffectiveAssetValue))
}	# function

# ***************************************************************************
# Simulate total returns for different share classes
# ***************************************************************************
# returns list(TotalFundReturn, FinalAssetValue)
SimulateTotalReturns  =  function(                        
    ParameterList,    # front-end load, fee rates, back-end load etc.
    Nmax,             # upper limit to periods for simulation (1,2, ... Nmax)
    MaxReturn,        # upper limit to assumed fixed returns (1,2,... MaxReturn)
    SharePrice,       # initial purchase price of a share
    MaxShareInd,      # #shares = [1,2,... MaxShareInd] * ShareMultiplier
    ShareMultiplier)
{
# number of share classes
CountShareClasses = length(ParameterList)

# initiate array to store results
TotalFundReturn = array(0, c(CountShareClasses, MaxShareInd, Nmax, MaxReturn))
FinalAssetValue = array(0, c(CountShareClasses, MaxShareInd, Nmax, MaxReturn))

	for (ShareClassNo in 1:CountShareClasses) {
		# read all inputs from InputParameters
		FrondEndLoadRates = ParameterList[[ShareClassNo]][[1]]
		BackEndLoadCDSCRates = ParameterList[[ShareClassNo]][[2]]
		AnnualDistributionFeeRate = ParameterList[[ShareClassNo]][[3]]
		AnnualManagementFeeRates = ParameterList[[ShareClassNo]][[4]]
		MngFeeScaleLogic = ParameterList[[ShareClassNo]][[5]]
		OtherAnnualServiceFeeRates = ParameterList[[ShareClassNo]][[6]]
		OtherSubscriptionFeeRates = ParameterList[[ShareClassNo]][[7]]
		OtherRedemptionFeeRates = ParameterList[[ShareClassNo]][[8]]

		for (IndShares in 1:MaxShareInd){
			NumberOfShares = IndShares * ShareMultiplier
			
			for (N in 1:Nmax) {
				for (AssumedFixedReturn in 1:MaxReturn) {
					AssumedFundReturns = AssumedFixedReturn * rep(1,N)
					
					# Call multi-period fee calculation function
					res = FeeCalculator_MultiPeriod(    
						FrondEndLoadRates,              
						BackEndLoadCDSCRates,           
						AnnualDistributionFeeRate,      
						AnnualManagementFeeRates,       
						MngFeeScaleLogic,               
						OtherAnnualServiceFeeRates,     
						OtherSubscriptionFeeRates,      
						OtherRedemptionFeeRates,        
						AssumedFundReturns,             
						SharePrice,       
						NumberOfShares,                   
						N)

					# ResultMatrix = res[[1]]
					# ResultMatrixFields = res[[2]]
					# EffectiveFundReturns = res[[3]]
					EffectiveAssetValue = res[[4]]
					
					# store results in multi-dimensional arrays
					TotalFundReturn[ShareClassNo,IndShares,N,AssumedFixedReturn] = ((EffectiveAssetValue[N] / (NumberOfShares * SharePrice))-1) * 100
					FinalAssetValue[ShareClassNo,IndShares,N,AssumedFixedReturn] = EffectiveAssetValue[N]
				}
			}
		}
	}
return(list(TotalFundReturn, FinalAssetValue))
}	# function
# ***************************************************************************
# Find optimal share class with max total return
# ***************************************************************************
FindOptimalShareClass = function(TotalFundReturn,IndShares,N,AssumedFixedReturn){
#total number of share classes
NumShareClases = dim(TotalFundReturn)[1]

TotalReturns = rep(0,NumShareClases)

	for (i in 1:NumShareClases) {
		TotalReturns[i] = TotalFundReturn[i,IndShares,N,AssumedFixedReturn]
	}

OptimShareClass = which.max(TotalReturns)
return(OptimShareClass)
} # function

# finaquant Financial Analytics - www.finaquant.com
# Tunc Ali Ktkcoglu  2012